﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2Request.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancingv2 {
namespace Model {

/**
 */
class DescribeListenersRequest : public ElasticLoadBalancingv2Request {
 public:
  AWS_ELASTICLOADBALANCINGV2_API DescribeListenersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeListeners"; }

  AWS_ELASTICLOADBALANCINGV2_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCINGV2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the load balancer.</p>
   */
  inline const Aws::String& GetLoadBalancerArn() const { return m_loadBalancerArn; }
  inline bool LoadBalancerArnHasBeenSet() const { return m_loadBalancerArnHasBeenSet; }
  template <typename LoadBalancerArnT = Aws::String>
  void SetLoadBalancerArn(LoadBalancerArnT&& value) {
    m_loadBalancerArnHasBeenSet = true;
    m_loadBalancerArn = std::forward<LoadBalancerArnT>(value);
  }
  template <typename LoadBalancerArnT = Aws::String>
  DescribeListenersRequest& WithLoadBalancerArn(LoadBalancerArnT&& value) {
    SetLoadBalancerArn(std::forward<LoadBalancerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Names (ARN) of the listeners.</p>
   */
  inline const Aws::Vector<Aws::String>& GetListenerArns() const { return m_listenerArns; }
  inline bool ListenerArnsHasBeenSet() const { return m_listenerArnsHasBeenSet; }
  template <typename ListenerArnsT = Aws::Vector<Aws::String>>
  void SetListenerArns(ListenerArnsT&& value) {
    m_listenerArnsHasBeenSet = true;
    m_listenerArns = std::forward<ListenerArnsT>(value);
  }
  template <typename ListenerArnsT = Aws::Vector<Aws::String>>
  DescribeListenersRequest& WithListenerArns(ListenerArnsT&& value) {
    SetListenerArns(std::forward<ListenerArnsT>(value));
    return *this;
  }
  template <typename ListenerArnsT = Aws::String>
  DescribeListenersRequest& AddListenerArns(ListenerArnsT&& value) {
    m_listenerArnsHasBeenSet = true;
    m_listenerArns.emplace_back(std::forward<ListenerArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The marker for the next set of results. (You received this marker from a
   * previous call.)</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeListenersRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with this call.</p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline DescribeListenersRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_loadBalancerArn;

  Aws::Vector<Aws::String> m_listenerArns;

  Aws::String m_marker;

  int m_pageSize{0};
  bool m_loadBalancerArnHasBeenSet = false;
  bool m_listenerArnsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancingv2
}  // namespace Aws
